import unittest
from datetime import date
from playgroup import GroupInfo, PlayGroup
from position_rank import Position, FilledPosition, Rank

class TestPlayGroup(unittest.TestCase):
    def setUp(self):
        self.groupinfo = GroupInfo("A", "Europe", "Phase 1", date(2019, 4, 8))
        class StubTeam(object):
            def __init__(self, name):
                self._name = name
            @property
            def name(self):
                return self._name
            def __eq__(self, other): #hmm toto som nespecifikoval ako sucast interfacu teamu. Bud by som to nemal pouzivat alebo napisat, ze team ma ==
                return self._name == other.name()
        self.germany = StubTeam("Germany")
        self.slovakia = StubTeam("Slovakia")
        self.france = StubTeam("France")
        self.rank_slovakia = Rank(True, FilledPosition(self.slovakia), {})
        self.rank_germany = Rank(False, FilledPosition(self.germany), {})
        self.positions = (Position(self.rank_slovakia),
                          Position(self.rank_germany),
                          FilledPosition(self.france))

        class StubMatch(object):
            def __init__(self, current_date, positions, future_data, test_case):
                self._position1 = positions[0]
                self._position2 = positions[1]
                self._test_case = test_case
                self._date = current_date
                self._future_data = future_data
                self._data = None
                self.times_played = 0
            @property
            def position1(self):
                return self._position1
            @property
            def position2(self):
                return self._position2
            @property
            def date(self): #som ovela viac safe ako vacsina rieseni, nedovolim kodu prepisat den hrania zapasu
                return self._date
            def play(self):
                self.times_played += 1 #kontrolujem ci sa zapas hra raz
                self._data = self._future_data
            @property
            def data(self):
                return self._data
        self.match1 = StubMatch(date(2019, 4, 3), (self.positions[1], self.positions[0]),
                                "Match1 Info", self)
        self.matches = (self.match1,)

        class MockRanking(object):
            @property
            def info_strings(self):
                return ("goals", )
            def rank(self, teams, matches): #kontrolujem, ci posielam tie informacie, ktore mam. Mozno je tu ale problem, ze tento test je az prilis striktny.
                self.check_teams(teams)
                self.check_matches(matches)
                return self.result

        self.ranking = MockRanking()
        self.ranking.result = ((0, {"goals": 0}), (1, {"goals": 0}), (2, {"goals": 0}))

        self.ranking.check_teams = lambda x: x == [self.slovakia.name, "???", self.france.name]
        self.ranking.check_matches = lambda x: x == []

        self.playgroup = PlayGroup(date(2019, 3, 31), self.groupinfo, self.positions,
                                   self.ranking, self.matches)

        self.ranking.check_teams = lambda x: True
        self.ranking.check_matches = lambda x: True

    def test_standing_entries(self):
        self.assertEqual(self.playgroup.standings_entries, ("goals", ))

    def test_initial_group_state(self):
        groupstate = self.playgroup.group_state(("goals",))
        self.assertEqual(groupstate.name, "A")
        self.assertEqual(groupstate.zone, "Europe")
        self.assertEqual(groupstate.phase, "Phase 1")
        self.assertEqual(len(groupstate.standings), 3)
        self.assertEqual(groupstate.standings[0], (self.slovakia.name, {"goals": 0}))
        self.assertEqual(groupstate.standings[1], ("???", {"goals": 0}))
        self.assertEqual(groupstate.standings[2], (self.france.name, {"goals": 0}))


    def test_positions_resolution_in_group_state(self):
        self.rank_germany.make_final()
        groupstate = self.playgroup.group_state(())
        self.assertEqual(groupstate.standings[0], (self.slovakia.name, {}))
        self.assertEqual(groupstate.standings[1][0], self.germany.name)

    def test_today_matches_when_match_is_resolved(self):
        self.rank_germany.make_final()
        self.playgroup.next_day()
        self.assertEqual(len(self.playgroup.today_matches), 0)
        self.playgroup.next_day()
        self.assertEqual(len(self.playgroup.today_matches), 1)
        self.assertEqual(self.playgroup.today_matches[0], (self.germany.name, self.slovakia.name))

    def test_today_matches_when_match_is_not_resolved(self):
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.assertEqual(len(self.playgroup.today_matches), 1)
        self.assertEqual(self.playgroup.today_matches[0], ("???", self.slovakia.name))

    def test_match_is_played_on_its_date(self):
        self.rank_germany.make_final()
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.assertIsNone(self.match1.data)
        self.playgroup.next_day()
        self.assertIsNotNone(self.match1.data)
        self.assertEqual(self.match1.times_played, 1)
        self.playgroup.next_day()
        self.assertEqual(self.match1.times_played, 1)

    def test_assert_failure_if_match_starts_when_some_position_is_unresolved(self):
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.assertRaises(AssertionError, self.playgroup.next_day)

    def test_after_match_is_played_teams_are_reordered(self):
        self.rank_germany.make_final()
        self.ranking.check_teams = lambda x: \
                 x == [self.slovakia.name, self.germany.name, self.france.name]
        self.ranking.check_matches = lambda x: x == ["Match1 Info"]
        self.ranking.result = ((1, {"goals": 2}), (0, {"goals": 1}), (2, {"goals": 0}))
        self.playgroup.next_day()
        self.playgroup.next_day()
        groupstate = self.playgroup.group_state(("goals",))
        self.assertEqual(groupstate.standings[0], (self.slovakia.name, {"goals": 0}))
        self.assertEqual(groupstate.standings[1], (self.germany.name, {"goals": 0}))
        self.assertEqual(groupstate.standings[2], (self.france.name, {"goals": 0}))
        self.playgroup.next_day()
        groupstate = self.playgroup.group_state(("goals",))
        self.assertEqual(groupstate.standings[0], (self.germany.name, {"goals": 1}))
        self.assertEqual(groupstate.standings[1], (self.slovakia.name, {"goals": 2}))
        self.assertEqual(groupstate.standings[2], (self.france.name, {"goals": 0}))

    def test_ranks_are_final_after_end_date(self):
        self.rank_germany.make_final()
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.playgroup.next_day()
        self.playgroup.next_day()
        #2019-4-8
        self.assertFalse(self.playgroup.rank(0).is_final())
        self.assertFalse(self.playgroup.rank(1).is_final())
        self.assertFalse(self.playgroup.rank(2).is_final())
        self.playgroup.next_day()
        #2019-4-9
        self.assertTrue(self.playgroup.rank(0).is_final())
        self.assertTrue(self.playgroup.rank(1).is_final())
        self.assertTrue(self.playgroup.rank(2).is_final())

if __name__ == '__main__':
    unittest.main()
